/***************************************************************
 *                 Mathematical Object Library                 *
 * class linear_systems : decl. for Linear Systems Resolution  *
 *                    simula.plus@cemes.fr                     *
 *                   GNU/linux version 3.2.0                   *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2003,2004,2005,2006,2008,2009,2012 COLLARD Christophe
 * copyright © 2003,2004 CREUSE Emmanuel
 * copyright © 2004 DJADEL Karim
 * copyright © 2003,2004,2005,2006,2008,2009,2012 Centre National de la Recherche Scientifique
 * copyright © 2003,2004,2005,2006,2008,2009 Arts et Métiers ParisTech
 * copyright © 2003,2004,2005,2006 Université de Valenciennes et du Hainaut Cambrésis
 * copyright © 2003,2004,2005,2006,2008,2009 Laboratoire de Physique et Mécanique des Matériaux (LPMM - CNRS)
 * copyright © 2003,2004,2005,2006 Laboratoire de Mathématiques et ses Applications de Valenciennes (LAMAV)
 * copyright © 2012 Centre d'Elaboration de Matériaux et d'Etudes Structurales (CEMES - CNRS)
 ***************************************************************/

/*! \namespace mol
    \brief Mathematical Object Libraries
*/

/*! \class mol::linear_systems
    \brief Library for linear systems resolution

    \htmlonly 
    <FONT color="#838383">

    linear_systems belongs to Mathematical Object Libraries (MOL++) </br>
    MOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \authors copyright \htmlonly &#169; \endhtmlonly 2003, 2004, 2005, 2006, 2008, 2009, 2012 Christophe COLLARD \n
             copyright \htmlonly &#169; \endhtmlonly 2003, 2004 Emmanuel CREUSE \n
	     copyright \htmlonly &#169; \endhtmlonly 2004 Karim DJADEL \n
             copyright \htmlonly &#169; 2003, 2004, 2005, 2006, 2008, 2009, 2012 Centre National de la Recherche Scientifique \endhtmlonly \n
	     copyright \htmlonly &#169; 2003, 2004, 2005, 2006, 2008, 2009 Arts et M&#233;tiers ParisTech \endhtmlonly \n
	     copyright \htmlonly &#169; 2003, 2004, 2005, 2006 Universit&#233; de Valenciennes et du Hainaut Cambr&#233;sis \endhtmlonly \n
	     copyright \htmlonly &#169; 2003, 2004, 2005, 2006, 2008, 2009 Laboratoire de Physique et M&#233;canique des Mat&#233;riaux (LPMM - CNRS) \endhtmlonly \n
	     copyright \htmlonly &#169; 2003, 2004, 2005, 2006 Laboratoire de Math&#233;matiques et ses Applications de Valenciennes (LAMAV) \endhtmlonly \n
	     copyright \htmlonly &#169; 2012 Centre d'Elaboration de Mat&#233;riaux et d'Etudes Structurales (CEMES - CNRS) \endhtmlonly \n
    \version linux 3.2.0
    \date 2003-2012
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type linear systems
#endif

#ifndef __linear_systems_hpp
#define __linear_systems_hpp


#ifndef __vectors_hpp
#include "vectors.hpp"
#endif

#ifndef __matrix_hpp
#include "matrix.hpp"
#endif

#ifndef __symmatrix_hpp
#include "symmatrix.hpp"
#endif

#ifndef __spmatrix_hpp
#include "spmatrix.hpp"
#endif

// WARNING : Do not include matrix.h here !

// contents :

using namespace std;

namespace mol
{


//==================
class linear_systems
//==================
{
  public :
    // Solves Ax=b by Gauss algorithm (pivot partiel)
    template <class T> friend vector<T> gauss (matrix<T>, vector<T>);

    // Solves Ax=b by Gauss algorithm (pivot total)
    template <class T> friend vector<T> gauss_pt (matrix<T>, vector<T>);

    // Solves Ax=b by gradient algorithm
    template <class matrix_type, class T>
      friend vector<T> gradient (const matrix_type&, const vector<T>&, const vector<T>&, int&, long double = precision);
    template <class matrix_type, class T>
      friend vector<T> gradient (const matrix_type&, const vector<T>&, const vector<T>&, long double = precision);
    
    // Solves Ax=b by conjugate gradient algorithm
    template <class matrix_type, class T>
      friend vector<T> conjugate_gradient (const matrix_type&, const vector<T>&, const vector<T>&, int&, long double = precision);
    template <class matrix_type, class T>
      friend vector<T> conjugate_gradient (const matrix_type&, const vector<T>&, const vector<T>&, long double = precision);

    // Methodes de preconditionnement de la matrice A : A = M-N = D-E-F
    // ----------------------------------------------------------------

    // Resolution de Ax=b par la methode de Jacobi :           M=D     N=E+F
    template <class matrix_type, class T>
      friend vector<T> jacobi (const matrix_type&, const vector<T>&, const vector<T>&, int&, long double = precision);
    template <class matrix_type, class T>
      friend vector<T> jacobi (const matrix_type&, const vector<T>&, const vector<T>&, long double = precision);

    // Resolution de Ax=b par la methode de Gauss-Seidel :     M=D-E   N=F
    template <class matrix_type, class T>
      friend vector<T> gauss_seidel (const matrix_type&, const vector<T>&, const vector<T>&, int&, long double = precision);
    template <class matrix_type, class T>
      friend vector<T> gauss_seidel (const matrix_type&, const vector<T>&, const vector<T>&, long double = precision);

    // Resolution de Ax=b par la methode de relaxation (SOR) : M=D-wE  N=(1-w)E+F
    template <class matrix_type, class T>
      friend vector<T> sor (const matrix_type&, const vector<T>&, const vector<T>&, T, int&, long double = precision);
    template <class matrix_type, class T>
      friend vector<T> sor (const matrix_type&, const vector<T>&, const vector<T>&, T, long double = precision);

    // Resolution of Ax=b by preconditionned gradient method (SSOR)   : M=1/w(2+w) * (D-wE) "1/D" (D-wF)
    template <class matrix_type, class T>
      friend vector<T> ssor (const matrix_type&, const vector<T>&, const vector<T>&, T, int&, long double = precision);
    template <class matrix_type, class T>
      friend vector<T> ssor (const matrix_type&, const vector<T>&, const vector<T>&, T, long double = precision);

    // ------------------------------------------------------------------------------------------------------------------------
    /*
      template <class T> friend vector<T> up_ssor (T, const vector<T>&, const matrix<T>&, const vector<T>& );
   			// compute the up with the triangular superior SSOR preconditionner
			// (for classes  " matrix " and " symmatrix " only)
      template <class T> friend vector<T> down_ssor (T, const vector<T>&, const matrix<T>&, const vector<T>& );
   			// compute the down with the triangular inferior SSOR preconditionner
			// (for classes  " matrix " and " symmatrix " only)
      template <class T> friend vector<T> up_ssor (T, const vector<T>&, const sparse_matrix<T>&, const vector<T>& );
   			// compute the up with the triangular superior SSOR preconditionner
			// (for classe  " sparse_matrix " only)
      template <class T> friend vector<T> down_ssor (T, const vector<T>&, const sparse_matrix<T>&, const vector<T>& );
   			// compute the down with the triangular inferior SSOR preconditionner
			// (for classe  " sparse_matrix " only)
      template <class matrix_type, class T>
        friend vector<T> ssor (T, const matrix_type&, const vector<T>&, const vector<T>&, long double&, int&);
  template <class matrix_type, class T>
    friend vector<T> ssor (const matrix_type&, const vector<T>&, const vector<T>&, long double&, int&);
  template <class matrix_type, class T>
    friend vector<T> ssor (T, const matrix_type&, const vector<T>&, const vector<T>&, T);
  template <class matrix_type, class T>
    friend vector<T> ssor (T, const matrix_type&, const vector<T>&, const vector<T>&, long double&);
  */

    // Resolution of Ax = b by BiConjuguate Gradient Stabilized
    template <class matrix_type, class T>
      friend vector<T> BICGSTAB (const matrix_type&, const vector<T>&, const vector<T>&, long double&, int&);
    template <class matrix_type, class T>
      friend vector<T> BICGSTAB (const matrix_type&, const vector<T>&, const vector<T>&, long double&);
    template <class matrix_type, class T>
      friend vector<T> BICGSTAB (const matrix_type&, const vector<T>&, const vector<T>&, int&);
};


//================================Resolution de Ax=b par la methode de Gauss==========================================================


//----------------------------------------------------------------
template <class T> vector<T> gauss(matrix<T> matrice, vector<T> b)
//----------------------------------------------------------------
{ // uncomment the following line to check if the matrix can be inverted - ! Be carefull : the computation time is widely increased !
  // assert(abs(det(matrice))>epsilon);
  assert (matrice.Rows()==b.dim());
  assert (matrice.Rows()==matrice.Columns()); // matrix must be squared

  int dim = b.dim();
  int pos;
  T pivot, elt;

  for (int k=1; k<=dim; k++)
    { pivot = 0;
      // search the biggest element of the kth column
      for (int i=k; i<=dim; i++)
        if (abs(matrice(i,k)) > abs(pivot))
            { pivot = matrice(i,k);
	      pos = i;
	    }
      // row k <-> row pos
      assert (abs(pivot)>epsilon);
      if (pos != k)
	{ matrice.permute_rows(k,pos);
	  b.permute(k,pos);
	}
      //  starts the resolution at step k
      for (int i=k+1; i<=dim; i++)
     	{ elt = matrice(i,k) / matrice(k,k);
	  for (int j=k+1; j<=dim; j++)
     	     matrice(i,j) -= matrice(k,j) * elt;
	  //	  matrice(i,k) = 0;
      	  b[i] -= b[k] * elt;
      	}
    }
  // algo de remontee
  for (int i=dim; i>0; i--)
    { for (int j=i+1; j<=dim; j++)
        b[i] -= b[j] * matrice(i,j);
      b[i] /= matrice(i,i);
    }
  return b;
}


//================================Resolution de Ax=b par la methode de Gauss Pivot Total==============================================


//-------------------------------------------------------------------
template <class T> vector<T> gauss_pt(matrix<T> matrice, vector<T> b)
//-------------------------------------------------------------------
{ // uncomment the following line to check if the matrix can be inverted - ! Be carefull : the computation time is widely increased !
  // assert(vasb(det(matrice))>epsilon);   
  assert(matrice.Rows()==b.dim());
  assert (matrice.Rows()==matrice.Columns()); // matrix must be squared

  vector<T> v(b.dim(),false);
  int dim = b.dim();
  int posx, posy;
  T pivot, elt;
  vector<int> u_coord(dim,false);

  for (int i=1; i<=dim; i++) u_coord[i] = i; // coordinates position of the unknow -> (u1,u2,...,un)

  for (int k=1; k<=dim; k++)
    { pivot = 0;

      // search the biggest element in the submatrix
      for (int i=k; i<=dim; i++)
	for (int j=k; j<=dim; j++)
	  if (abs(matrice(i,j)) > abs(pivot))
	    { pivot = matrice(i,j);
	      posx = i;
	      posy = j;
	    }
      assert (abs(pivot)>epsilon);

      // Rows permutation : row k <-> row posx
      if (posx != k)
	{ matrice.permute_rows(k,posx);
	  b.permute(k,posx);
	}

      // Columns permutation : column k <-> column posy
      if (posy!=k)
	{ u_coord.permute(k,posy);
	  matrice.permute_columns(k,posy);
	}

      //  starts the resolution at step k
      for (int i=k+1; i<=dim; i++)
     	{ elt = matrice(i,k) / matrice(k,k);
          for (int j=k+1; j<=dim; j++)
	    matrice(i,j) -= matrice(k,j) * elt;
      	  b[i] -= b[k] * elt;
      	}
    }

  // algo de remontee
  for (int i=dim; i>0; i--)
    { for (int j=i+1; j<=dim; j++)
	b[i] -= b[j] * matrice(i,j);
    b[i] /= matrice(i,i);
    }

  // reorganisation du vecteur
  for (int i=1; i<=dim; i++)
    v[u_coord[i]] = b[i];

  return v;
}


//================================Resolution de Ax=b par la methode du gradient=======================================================


//------------------------------------------------------------------------------------------------------------------------
template <class matrix_type, class T>
vector<T> gradient (const matrix_type& matrice, const vector<T>& b, const vector<T>& x0, int& iterations, long double eps)
//------------------------------------------------------------------------------------------------------------------------
{
  assert (abs(b.norm())>epsilon);
  assert (eps > 0);

  vector<T> x = x0, g, Aw;
  T rho;
  g = matrice * x0 - b;
  iterations = 0;

  for (; g.norm()>eps; iterations++)
    {
      Aw = (matrice * g);
      rho = - (g * g) / (Aw * g);
      x += rho * g;
      g += rho * Aw;
    }

  return x;
}


//-------------------------------------------------------------------------------------------------------
template <class matrix_type, class T>
vector<T> gradient (const matrix_type& matrice, const vector<T>& b, const vector<T>& x0, long double eps)
//-------------------------------------------------------------------------------------------------------
{
  int iter;
  return gradient (matrice, b, x0, iter, eps);
}


//================================Resolution de Ax=b par la methode du gradient conjugue==============================================


//----------------------------------------------------------------------------------------------------------------------------------
template <class matrix_type, class T>
vector<T> conjugate_gradient (const matrix_type& matrice, const vector<T>& b, const vector<T>& x0, int& iterations, long double eps)
//----------------------------------------------------------------------------------------------------------------------------------
{
  assert (abs(b.norm())>epsilon);
  assert (eps > 0);

  vector<T> x = x0, g, w, Aw;
  T rho, Aww;
  w = g = matrice * x0 - b;
  iterations = 0;

  for (; g.norm()>eps; iterations++)
    { Aw = matrice * w;
      Aww = Aw*w;
      rho = - (g * w) / (Aww);
      x += rho * w;
      g += rho * Aw;
      w = g - (g * Aw) / (Aww) * w;
    }

  return x;
}


//-----------------------------------------------------------------------------------------------------------------
template <class matrix_type, class T>
vector<T> conjugate_gradient (const matrix_type& matrice, const vector<T>& b, const vector<T>& x0, long double eps)
//-----------------------------------------------------------------------------------------------------------------
{
  int iter;
  return conjugate_gradient (matrice,b,x0,iter,eps);
}


//================================Resolution de Ax=b par la methode de Jacobi=========================================================

/*!
  \brief Solves Ax=b with the Jacobi algorithm.

  We split A in the following way : \n
  A = M-N = D-E-F (D=diagonal E=lower F=upper) \n
  with \f$ \displaystyle M = D \f$ \n
  and  \f$ \displaystyle N = E + F \f$. \n
  So we compute
  \f$ \displaystyle D u^{n+1} = \left( E + F \right)  u^n + b
      \Leftrightarrow
      \boxed{u^{n+1}  = D^{-1} \left[ \left( E + F \right) u^n + b \right] } \f$
  \param matrice A matrix
  \param b b vector
  \param u0 starting point for the algorithm
  \param eps precision used for convergence test
  \param iteration number of iterations for the sor algorithm
  \return u linear system solution
*/

//----------------------------------------------------------------------------------------------------------------------
template <class matrix_type, class T>
vector<T> jacobi (const matrix_type& matrice, const vector<T>& b, const vector<T>& u0, int& iterations, long double eps)
//----------------------------------------------------------------------------------------------------------------------
{
  // A = M-N = D-E-F (D=diagonal E=lower F=upper)
  // with M = D
  // and  N = E+F
  assert (matrice.Rows() == b.dim());
  assert (abs(b.norm())>epsilon);
  assert (eps > 0);

  vector<T> u = u0, v = u0;
  iterations = 0;

  for (; (matrice*u-b).norm() / b.norm() > eps; u=v, iterations++) // u^n+1 = v
  for (int i=1; i<= matrice.Rows(); i++)
  {   v[i] = b[i];                                 // v = b
      for (int j=1; j<i;j++)
	  v[i] -= matrice(i,j) * u[j];             // v -= E u^n  =>  v = b + E u^n
      for (int j=i+1; j<= matrice.Columns(); j++)
	  v[i] -= matrice(i,j) * u[j];             // v -= F u^n  =>  v =  b + E u^n + F u^n
      v[i] /= matrice(i,i);                        // v /= D      =>  v = "1/D" [ b + E u^n + F u^n ]
  }

  return v;
}


//-----------------------------------------------------------------------------------------------------
template <class matrix_type, class T>
vector<T> jacobi (const matrix_type& matrice, const vector<T>& b, const vector<T>& u0, long double eps)
//-----------------------------------------------------------------------------------------------------
{
  int iter;
  return jacobi(matrice,b,u0,iter,eps);
}


//================================Resolution de Ax=b par la methode de Gauss Seidel===================================================

/*!
  \brief Solves Ax=b with the Gauss Seidel algorithm.

  We split A in the following way : \n
  A = M-N = D-E-F (D=diagonal E=lower F=upper) \n
  with \f$ \displaystyle M = D - E \f$ \n
  and  \f$ \displaystyle N = F \f$. \n
  So we compute
  \f$ \displaystyle \left( D - E \right) u^{n+1} = F  u^n + b
      \Leftrightarrow
      \boxed{u^{n+1}  = D^{-1} \left[ E u^{n+1} + F u^n + b \right] } \f$
  \param matrice A matrix
  \param b b vector
  \param u0 starting point for the algorithm
  \param eps precision used for convergence test
  \param iteration number of iterations for the sor algorithm
  \return u linear system solution
*/

//----------------------------------------------------------------------------------------------------------------------------
template <class matrix_type, class T>
vector<T> gauss_seidel (const matrix_type& matrice, const vector<T>& b, const vector<T>& u0, int& iterations, long double eps)
//----------------------------------------------------------------------------------------------------------------------------
{
  // A = M-N = D-E-F (D=diagonal E=lower F=upper)
  // with M = D-E
  // and  N = F
  assert (matrice.Rows() == b.dim());
  assert (abs(b.norm())>epsilon);
  assert (eps > 0);

  vector<T> u = u0;
  iterations = 0;

  for (; (matrice*u-b).norm() / b.norm() > eps; iterations++) // precision test
  for (int i=1; i<=matrice.Rows(); i++)
    { u[i] = b[i];                                // u^(n+1) = b
      for (int j=1; j<i;j++)
	u[i] -= matrice(i,j) * u[j];              // E u^(n+1)  =>  u^(n+1) =  b + E u^(n+1)
      for (int j=i+1; j<=matrice.Columns(); j++)
	u[i] -= matrice(i,j) * u[j];              // F u^n  =>  u^(n+1) = b + E u^(n+1) + F u^n
      u[i] /= matrice(i,i);                       // "1/D"  =>  u^(n+1) = "1/D" [b + E u^(n+1) + F u^n]
  }

  return u;
}


//-----------------------------------------------------------------------------------------------------------
template <class matrix_type, class T>
vector<T> gauss_seidel (const matrix_type& matrice, const vector<T>& b, const vector<T>& u0, long double eps)
//-----------------------------------------------------------------------------------------------------------
{
  int iter;
  return gauss_seidel(matrice,b,u0,iter,eps);
}


//================================Resolution de Ax=b par la methode de Relaxation - SOR===============================================

/*!
  \brief Solves Ax=b with the SOR (successive over-relaxation) algorithm.

  We split A in the following way : \n
  A = M-N = D-E-F (D=diagonal E=lower F=upper) \n
  with \f$ \displaystyle M = \frac{D}{\omega} - E \f$ \n
  and  \f$ \displaystyle N = \frac{(1-\omega)}{\omega} D + F \f$. \n
  So we compute
  \f$ \displaystyle \left( \frac{D}{\omega} - E \right) u^{n+1} = \left( \frac{1-\omega}{\omega} D + F \right) u^n + b
      \Leftrightarrow
      \boxed{u^{n+1}  = \omega D^{-1} \left[ E u^{n+1} + \left( \frac{1-\omega}{\omega} D + F \right) u^n + b \right] } \f$
  \param matrice A matrix
  \param b b vector
  \param u0 starting point for the algorithm
  \param w \f$ \omega \f$
  \param eps precision used for convergence test
  \param iteration number of iterations for the sor algorithm
  \return u linear system solution
*/

//------------------------------------------------------------------------------------------------------------------------
template <class matrix_type, class T>
vector<T> sor (const matrix_type& matrice, const vector<T>& b, const vector<T>& u0, T w, int& iterations, long double eps)
//------------------------------------------------------------------------------------------------------------------------
{
  // A = M-N = D-E-F (D=diagonal E=lower F=upper)
  // with M = D/w-E
  // and  N = (1-w)D/w+F
  assert (matrice.Rows() == matrice.Columns());
  assert (matrice.Rows() == b.dim());
  assert (abs(b.norm())>epsilon);
  assert (abs(w)>epsilon);
  assert (eps > 0);

  // pre computations

  vector<T> wD =& (diagonal(matrice)/w); // wD = D / w
  vector<T> wDinv(matrice.Rows(),false);
  for (int i=1; i<=matrice.Rows(); i++)
    wDinv[i] = 1/wD[i];  // wDinv = "w / D"
  wD *= (1-w); // wD = (1-w)/w D

  vector<T> u = u0;
  iterations = 0;

  // starts resolution

  for (; (matrice*u-b).norm() / b.norm() > eps; iterations++) // precision test
    for (int i=1; i<=matrice.Rows(); i++)
      { u[i] *= wD[i];                              // u^(n+1)_i = (1-w)/w D u^n_i
	u[i] += b[i];                               // u^(n+1) = b + (1-w)/w D u^n
	for (int j=1; j<i;j++)
	  u[i] -= matrice(i,j) * u[j];              // E u^(n+1)  =>  u^(n+1) =  b + E u^(n+1) + (1-w)/w D u^n
	for (int j=i+1; j<=matrice.Columns(); j++)
	  u[i] -= matrice(i,j) * u[j];              // F u^n  =>  u^(n+1) = b + E u^(n+1) + [ (1-w)/w D + F ] u^n
	u[i] *= wDinv[i];                           // "w/D"  =>  u^(n+1) = "w/D" { b + E u^(n+1) + [ (1-w)/w D + F ] u^n }
      }

  return u;
}


//-------------------------------------------------------------------------------------------------------
template <class matrix_type, class T>
vector<T> sor (const matrix_type& matrice, const vector<T>& b, const vector<T>& u0, T w, long double eps)
//-------------------------------------------------------------------------------------------------------
{
  int iter;
  return sor(matrice,b,u0,w,iter,eps);
}


//================================Resolution de Ax=b par la methode de Relaxation symetrique - SSOR===================================

/*!
  \brief Solves Ax=b with the SSOR (symmetric successive over-relaxation) algorithm.

  We split A in the following way : \n
  A = M-N = D-E-F (D=diagonal E=lower F=upper) \n
  with \f$ \displaystyle M = \frac{1}{\omega (2 - \omega)} \left( D - \omega E \right) D^{-1} \left( D - \omega F \right) \f$. \n
  The computations for this algorithm are divided in two parts : \n
  first we compute \f$ \displaystyle \left( \frac{D}{w} - E \right) v^n = \left( \frac{1-\omega}{\omega} D + F \right) u^n + b \Leftrightarrow \boxed{ v^n = \omega D^{-1} \left[ E v^n + \left( \frac{1-\omega}{\omega} D + F \right) u^n + b \right] } \f$ \n
  and then we compute
   \f$ \displaystyle \left( \frac{D}{w} - F \right) u^{n+1} = \left( \frac{1-\omega}{\omega} D + E \right) v^n + b \Leftrightarrow \boxed{ u^{n+1} = \omega D^{-1} \left[ F u^{n+1} + \left( \frac{1-\omega}{\omega} D + E \right) v^n + b \right] } \f$
  \param matrice A matrix
  \param b b vector
  \param u0 starting point for the algorithm
  \param w \f$ \omega \f$
  \param eps precision used for convergence test
  \param iteration number of iterations for the sor algorithm
  \return u linear system solution
*/

//-------------------------------------------------------------------------------------------------------------------------
template <class matrix_type, class T>
vector<T> ssor (const matrix_type& matrice, const vector<T>& b, const vector<T>& u0, T w, int& iterations, long double eps)
//-------------------------------------------------------------------------------------------------------------------------
{
  // A = M-N = D-E-F (D=diagonal E=lower F=upper)
  // with M = 1/[w(2-w)] (D-wE)"1/D"(D-wF)
  assert (matrice.Rows() == matrice.Columns());
  assert (matrice.Rows() == b.dim());
  assert (abs(b.norm())>epsilon);
  assert (abs(w)>epsilon);
  assert (eps > 0);

  // pre computations

  vector<T> wD =& (diagonal(matrice)/w); // wD = D / w
  vector<T> wDinv(matrice.Rows(),false);
  for (int i=1; i<=matrice.Rows(); i++)
    wDinv[i] = 1/wD[i];  // wDinv = "w / D"
  wD *= (1-w); // wD = (1-w)/w D

  vector<T> u = u0;
  vector<T> v(u0.dim(),false);
  iterations = 0;

  // starts resolution

  for (; (matrice*u-b).norm() / b.norm() > eps; iterations++) // precision test
    {
      for (int i=1; i<=matrice.Rows(); i++)
	{ v[i] = wD[i] * u[i];                        // v^n = (1-w)/w D u^n
	  for (int j=i+1; j<=matrice.Columns(); j++)
	    v[i] -= matrice(i,j) * u[j];              // F u^n => v^n = [(1-w)/w D + F] u^n
	  for (int j=1; j<i; j++)
	    v[i] -= matrice(i,j) * v[j];              // E v^n => v^n = E v^n + [(1-w)/w D + F] u^n
	  v[i] += b[i];                               // b     => v^n = E v^n + [(1-w)/w D + F] u^n + b
	  v[i] *= wDinv[i];                           // "w/D" => v^n = "w/D" {E v^n + [(1-w)/w D + F] u^n + b}
	}

      for (int i=matrice.Rows(); i>0; i--)
	{ u[i] = wD[i] * v[i];                        // u^(n+1) = (1-w)/w D v^n
	  for (int j=1; j<i; j++)
	    u[i] -= matrice(i,j) * v[j];              // E v^n     => u^(n+1) = [(1-w)/w D + E] v^n
	  for (int j=i+1; j<=matrice.Columns(); j++)
	    u[i] -= matrice(i,j) * u[j];              // F u^(n+1) => u^(n+1) = F u^(n+1) + [(1-w)/w D + E] v^n
	  u[i] += b[i];                               // u^(n+1) = F u^(n+1) + [(1-w)/w D + E] v^n + b
	  u[i] *= wDinv[i];                           // u^(n+1) = "w/D" {F u^(n+1) + [(1-w)/w D + E] v^n + b}
	}
    }

  return u;
}


//--------------------------------------------------------------------------------------------------------
template <class matrix_type, class T>
vector<T> ssor (const matrix_type& matrice, const vector<T>& b, const vector<T>& u0, T w, long double eps)
//--------------------------------------------------------------------------------------------------------
{
  int iter;
  return ssor(matrice,b,u0,w,iter,eps);
}



/*

//------------------------------------------------------------------------------
// This function is useful for the SSOR
//  method : At each step of the algorithm,
//  we need to compute a up (and a down).
//  For the up, the vector "diagonal"
//  is used to store the diagonal of the matrix
//  omega belongs to ] 0, 2[
//  ----------> for "matrix " and "symmatrix "                
template <class T> vector<T> up_ssor(T omega, const vector<T>& diagonal,
                              const matrix<T>& A, const vector<T>& b)
//------------------------------------------------------------------------------
{
	vector<T> x(b.dim()) ;
	T temp ;
	T parameter = sqrt(omega*(2.-omega)) ;
	for ( int i = b.dim() ; i >= 1 ; i-- )
		{
			temp = 0. ;
	                for (int j=i+1;j<=b.dim();j++)
			  temp -= x[j] * A(i,j) ;
			x[i] = (temp * omega / sqrt(diagonal[i]) + parameter * b[i]) / sqrt(diagonal[i])  ;
		};
	return x ;
}


//------------------------------------------------------------------------------
// This function is useful for the SSOR
//  method : At each step of the algorithm,
//  we need to compute a down (and a up).
//  For the down, the vector "diagonal"
//  is used to store the diagonal of the matrix .
//  omega belongs to ] 0, 2[
//  ----------> for "matrix " and "symmatrix "           
template <class T> vector<T> down_ssor(T omega, const vector<T>& diagonal,
                              const matrix<T>& A, const vector<T>& b)
//------------------------------------------------------------------------------
{
	vector<T> x(b.dim()) ;
	T temp ;
	T parameter = sqrt(omega*(2.-omega)) ;
	for (int i=1;i<=b.dim();i++)
		{
			temp = 0.;
			for ( int j = 1 ; j < i ; j++ )
				temp -= x[j] * A(i,j) / sqrt(diagonal[j]) ;
			x[i] = (temp * omega + parameter * b[i]) / sqrt(diagonal[i]) ;
		} ;
	return x ;
}






//------------------------------------------------------------------------------
/ This function is useful for the SSOR
  method : At each step of the algorithm,
  we need to compute a up (and a down).
  For the up, the vector "diagonal"
  is used to store the diagonal of the matrix
  omega belongs to ] 0, 2[
  ----------> for " sparse_matrix " /
template <class T> vector<T> up_ssor(T omega, const vector<T>& diagonal,
					const sparse_matrix<T>& A, const vector<T>& b)
//------------------------------------------------------------------------------
{

	vector<T> x(b.dim()) ;
	T temp ;
	T parameter = sqrt(omega * (2. - omega)) ;
	for (int i = b.dim() ; i >= 1 ; i --)
		{
			temp = 0.;
			for ( int j = 1 ; j <= A.Row_Elements() ; j++ )
				if (A.partial_indice(i,j) > 0)
					if (A.partial_indice(i,j) > i)
						temp -= x[A.partial_indice(i,j)]
						* A.partial_value(i,j) ;

			x[i] = temp * omega / diagonal[i] + parameter * b[i] /
			sqrt(diagonal[i]) ;
		} ;
	return x ;
}


//------------------------------------------------------------------------------
/ This function is useful for the SSOR
  method : At each step of the algorithm,
  we need to compute a down (and a up).
  For the down, the vector "diagonal"
  is used to store the diagonal of the matrix .
  omega belongs to ] 0, 2[
  ----------> for " sparse_matrix "         /
template <class T> vector<T> down_ssor(T omega, const vector<T>& diagonal,
                              const sparse_matrix<T>& A, const vector<T>& b)
//------------------------------------------------------------------------------
{
	vector<T> x(b.dim()) ;
	T temp ;
	T parameter = sqrt(omega * (2. - omega)) ;
	for (int i = 1 ; i <= b.dim() ; i ++)
		{
			temp = 0.;
			for ( int j = 1 ; j <= A.Row_Elements() ; j++ )
				if (A.partial_indice(i,j) > 0)
					if (A.partial_indice(i,j) < i)
						temp -= x[A.partial_indice(i,j)]
						* A.partial_value(i,j)
						/ sqrt(diagonal[A.partial_indice(i,j)]) ;

			x[i] = (temp * omega + parameter * b[i]) / sqrt(diagonal[i]) ;
		} ;
	return x ;
}





//--------------------------------------------------------------------------------------------------------------------
template <class matrix_type, class M>
vector<M> ssor (M omega, const matrix_type& matrice, const vector<M>& b,
                const vector<M>& x0, long double& eps,
                int& iterations)
//--------------------------------------------------------------------------------------------------------------------
{
  vector<M> x(b.dim()),r0(b.dim()),p(b.dim()),
            z0(b.dim()),q(b.dim()),diag(b.dim()) ;
  x=x0;
  int i=0;

  diag = diagonal(matrice);

  M alpha,beta = 1.,temp,param ;
  param = b.norm() * eps;
  r0 = b - matrice * x;

  p = up_ssor(omega,diag,matrice,down_ssor(omega,diag,matrice,r0)) ;
  z0 = p ;
  temp = r0 * z0 ;


  while ( ( r0.norm() > param ) && ( i <= iterations ) )
  {

	q = matrice * p ;
	temp *= beta ;
	alpha = temp / ( q * p ) ;
	x += alpha * p ;
	r0 -= alpha * q  ;
	z0 = up_ssor(omega,diag,matrice,down_ssor(omega,diag,matrice,r0))  ;
	beta = (r0 * z0) / temp ;
	p = z0 + beta * p ;
	i += 1 ;

  };

  iterations = i ;
  eps = r0.norm() ;

  return x;
}

/*
// A = M-N = D-E-F (D=diagonal E=lower F=upper)
// with M = 1/w(2-w) (D-wE) "1/D" (D-wF)
//--------------------------------------------------------------------------------------------------------------------
template <class matrix_type, class T>
vector<T> ssor (T omega, const matrix_type& matrice, const vector<T>& b,
                const vector<T>& x0, long double& eps,
                int& iterations)
//--------------------------------------------------------------------------------------------------------------------
{
  vector<T> x(b.dim()),r0(b.dim()),p(b.dim()),
            z0(b.dim()),q(b.dim()),diagonal(b.dim()) ;
  x=x0;
  int i=0;

  diagonal = matrice.diagonal() ;

  T alpha,beta = 1.,temp,param ;
  param = b.norm() * eps;
  r0 = b - matrice * x;

  p = up_ssor(omega,diagonal,matrice,down_ssor(omega,diagonal,matrice,r0)) ;
  z0 = p ;
  temp = r0 * z0 ;


  //  while ( ( r0.norm() > param ) && ( i <= iterations ) )
  while ((matrice*x-b).norm() / b.norm() > eps)
  {

	q = matrice * p ;
	temp *= beta ;
	alpha = temp / ( q * p ) ;
	x += alpha * p ;
	r0 -= alpha * q  ;
	z0 = up_ssor(omega,diagonal,matrice,down_ssor(omega,diagonal,matrice,r0))  ;
	beta = (r0 * z0) / temp ;
	p = z0 + beta * p ;
	i += 1 ;

  };

  iterations = i ;
  eps = r0.norm() ;

  return x;
}
*/

 /*

//--------------------------------------------------------------------------------------------------------------------
template <class matrix_type, class T>
vector<T> ssor (const matrix_type& matrice, const vector<T>& b, const vector<T>& x0, long double& eps, int& iter)
//--------------------------------------------------------------------------------------------------------------------
{  // default value for omega
  T omega = 1. ;
  return ssor(omega,matrice,b,x0,eps,iter);

}


//--------------------------------------------------------------------------------------------------------------------
template <class matrix_type, class T>
vector<T> ssor (T omega, const matrix_type& matrice, const vector<T>& b, const vector<T>& x0, long double eps)
//--------------------------------------------------------------------------------------------------------------------
{
  int iter = b.dim() ;
  return ssor(omega,matrice,b,x0,eps,iter);

}



//--------------------------------------------------------------------------------------------------------------------
template <class matrix_type, class T>
vector<T> ssor (T omega, const matrix_type& matrice, const vector<T>& b, const vector<T>& x0, int& iter)
//--------------------------------------------------------------------------------------------------------------------
{
  long double eps = epsilon ;
  return ssor(omega,matrice,b,x0,eps,iter);

}

 */

//================================Resolution of Ax=b by Biconjugate Gradient Stalibized (BICGSTAB)==================================


//--------------------------------------------------------------------------------------------------------------------
template <class matrix_type, class T>
vector<T> BICGSTAB  (const matrix_type& matrice, const vector<T>& b, const vector<T>& x0,long double& eps, int& iterations)
//--------------------------------------------------------------------------------------------------------------------
{
  vector<T> x(b.dim()),r(b.dim()),r1(b.dim()),arb(b.dim()),as(b.dim()),p(b.dim()),ap(b.dim()),s(b.dim());
  x = x0 ;
  int i = 0 ;

  T alpha,omega,beta,temp1,temp2,param = b.norm() * eps ;
  r = b-matrice * x ;
  p = r ;
  arb = b ;  // arb is arbitrary vector in the method
  temp1 = r * arb ;

  while ( ( r.norm() > param ) && ( i <= iterations ) )
  {
    ap = matrice * p ;
    alpha = temp1 / (ap * arb) ;
    s = r - alpha * ap ;
    as = matrice * s ;
    omega =(as * s) / pow(as.norm(),2) ;
    x += alpha * p + omega * s ;
    r1 = s - omega * as ;
    temp2 = r1 * arb ;
    beta = (temp2 * alpha) / (temp1 * omega) ;
    p = r1 + beta *(p - omega * ap) ;
    r = r1;
    temp1 = temp2;
    i += 1 ;

  } ;

  iterations = i;
  eps = r.norm();
  return x;
}



//--------------------------------------------------------------------------------------------------------------------
template <class matrix_type, class T>
vector<T> BICGSTAB (const matrix_type& matrice, const vector<T>& b, const vector<T>& x0, long double eps)
//--------------------------------------------------------------------------------------------------------------------
{
  int iter = b.dim() ;
  return BICGSTAB(matrice,b,x0,eps,iter);

}


//--------------------------------------------------------------------------------------------------------------------
template <class matrix_type, class T>
vector<T> BICGSTAB (const matrix_type& matrice, const vector<T>& b, const vector<T>& x0, int& iter)
//--------------------------------------------------------------------------------------------------------------------
{
  long double eps = epsilon ;
  return BICGSTAB(matrice,b,x0,eps,iter);

}


}


#endif
